/*
 * file name: Relay_Secure_TX_Toggle_Simulation
 *
 * Description:
 * This code automatically simulates button presses to test the toggle logic.
 * - It simulates a press to send "TOG-ON".
 * - Waits for 10 seconds.
 * - Simulates another press to send "TOG-OFF".
 * - Waits for 5 seconds.
 * - The cycle repeats.
 *
 * This version uses a non-blocking timer to ensure radio events are processed.
 */
#include <Wire.h>
#include <HT_SSD1306Wire.h>
#include "LoRaWan_APP.h"
#include <Robojax_HeltecLoRa32.h>

// --- Configuration ---
bool debug = true; // Set to true to see Serial Monitor messages

// --- Display & Payload Text ---
const char *displayTexttitle = "Relay Sim:";
const char *displayTextTX = "(TX)";
const char *displayTextRelayToggleON = "TOG-ON";
const char *displayTextRelayToggleOFF = "TOG-OFF";

// --- LoRa & Security Settings ---
const char *userKey = "6tfDs$wEq3!"; // Security key. Must match the receiver.
#define RF_FREQUENCY 915555000 // Hz. Must match the frequency of the receiver.
#define TX_OUTPUT_POWER 20     // dBm from 2 to 20. (14 is a safe max on battery)

// --- LoRa Radio Parameters ---
#define LORA_BANDWIDTH 0         // [0: 125 kHz]
#define LORA_SPREADING_FACTOR 7  // [SF7..SF12]
#define LORA_CODINGRATE 1        // [1: 4/5]
#define LORA_PREAMBLE_LENGTH 8   // Same for Tx and Rx
#define LORA_FIX_LENGTH_PAYLOAD_ON false
#define LORA_IQ_INVERSION_ON false

// --- Global Variables ---
SSD1306Wire oledDisplay(0x3c, 500000, SDA_OLED, SCL_OLED, GEOMETRY_128_64, RST_OLED);
Robojax_HeltecLoRa32 robojaxDisplay(&oledDisplay);

// This variable is required by the Robojax_HeltecLoRa32 library
bool lora_idle = true;

// State management for toggle switch
String displayTextStateValue = displayTextRelayToggleOFF;
bool toggleState = false; // false = OFF, true = ON

// --- Variables for the simulation state machine ---
unsigned long previousMillis = 0;
int testState = 0; // 0: send ON, 1: wait 15s, 2: send OFF, 3: wait 5s
const long onInterval = 10000; // 15 seconds
const long offInterval = 5000; // 5 seconds


// --- Radio Event Functions ---
void OnTxDone(void) {
  if (debug) Serial.println("TX done.");
  lora_idle = true;
}

void OnTxTimeout(void) {
  Radio.Sleep(); // Put radio in low-power mode
  if (debug) Serial.println("TX Timeout.");
  lora_idle = true;
}

static RadioEvents_t RadioEvents;

// --- Helper function to simulate a single press and transmit ---
void simulatePressAndTransmit() {
  if (debug) Serial.println("---------------------------------");
  if (debug) Serial.println("Simulating button press...");
  
  // Flip the toggle state
  toggleState = !toggleState;
  
  // Update the text to be displayed and sent
  if (toggleState) {
    displayTextStateValue = displayTextRelayToggleON;
  } else {
    displayTextStateValue = displayTextRelayToggleOFF;
  }
  
  // Update the display immediately to show the new state
  robojaxDisplay.displayLineText(displayTexttitle, 0, 0, 24, false);
  robojaxDisplay.displayText(displayTextTX, 127, 5, 10, TEXT_ALIGN_RIGHT);
  robojaxDisplay.displayLineText(displayTextStateValue, 0, 30, 24, true);
  
  if (debug) Serial.printf("New state: %s. Transmitting...\n", displayTextStateValue.c_str());
  
  // Transmit the new state
  robojaxDisplay.sendStringSecure(displayTextStateValue);
  
  if (debug) Serial.println("Transmission call finished.");
}


// --- Setup Function ---
void setup() {
  Serial.begin(115200);
  
  // Power on external components
  pinMode(Vext, OUTPUT);
  digitalWrite(Vext, LOW);
  delay(100);

  // Initialize OLED display
  robojaxDisplay.begin();
  
  // Initialize LoRa Radio
  Mcu.begin(HELTEC_BOARD, SLOW_CLK_TPYE);
  RadioEvents.TxDone = OnTxDone;
  RadioEvents.TxTimeout = OnTxTimeout;
  Radio.Init(&RadioEvents);
  Radio.SetChannel(RF_FREQUENCY);
  Radio.SetTxConfig(MODEM_LORA, TX_OUTPUT_POWER, 0, LORA_BANDWIDTH,
                    LORA_SPREADING_FACTOR, LORA_CODINGRATE,
                    LORA_PREAMBLE_LENGTH, LORA_FIX_LENGTH_PAYLOAD_ON,
                    true, 0, 0, LORA_IQ_INVERSION_ON, 3000);
  
  if (debug) Serial.println("Toggle TX Simulation Initialized.");
  if (debug) Serial.println("Starting test sequence...");
}

// --- Main Loop ---
void loop() {
  unsigned long currentMillis = millis();

  // This state machine runs the test sequence without using delay()
  switch (testState) {
    case 0: // State 0: Send the ON signal
      simulatePressAndTransmit();
      previousMillis = currentMillis; // Reset the timer
      testState = 1; // Move to the next state (wait)
      if (debug) Serial.printf("Waiting for %ld seconds...\n", onInterval / 1000);
      break;
      
    case 1: // State 1: Wait for the ON interval to pass
      if (currentMillis - previousMillis >= onInterval) {
        testState = 2; // Time's up, move to the next state
      }
      break;

    case 2: // State 2: Send the OFF signal
      simulatePressAndTransmit();
      previousMillis = currentMillis; // Reset the timer
      testState = 3; // Move to the next state (wait)
      if (debug) Serial.printf("Waiting for %ld seconds...\n", offInterval / 1000);
      break;
      
    case 3: // State 3: Wait for the OFF interval to pass
      if (currentMillis - previousMillis >= offInterval) {
        testState = 0; // Time's up, repeat the sequence
      }
      break;
  }

  // CRITICAL: This must be called continuously to process radio events.
  // Using delay() prevents this from running, which was the cause of the bug.
  Radio.IrqProcess(); 
}
