/*
 * file name: Relay_Secure_TX_Toggle_Polling
 * *written on July 03, 2025 by Ahamd Shamshiri www.Robojax.com
 * Description:
 * This code uses a polling method (no interrupts) to read the button,
 * which is more robust for debugging complex issues.
 * - Each press of the push button (pin 0) toggles the state.
 * - It sends "TOG-ON" on the first press, "TOG-OFF" on the second, and so on.
 * - This uses the onboard 10k pull-up resistor, so the button should connect pin 0 to GND.
 * * Watch Full video instruction: https://youtu.be/4QZlciNqiDU
 * 📚⬇️ Download and resource page https://robojax.com/T641
 */
#include <Wire.h>
#include <HT_SSD1306Wire.h>
#include "LoRaWan_APP.h"
#include <Robojax_HeltecLoRa32.h>

// --- Configuration ---
bool debug = true; // Set to true to see Serial Monitor messages
#define PUSH_BUTTON_PIN 0  // The pin connected to the push button

// --- Display & Payload Text ---
const char *displayTextTitle = "Relay(T):";
const char *displayTextTX = "(TX)";
const char *displayTextRelayToggleON = "TOG-ON";
const char *displayTextRelayToggleOFF = "TOG-OFF";

// --- LoRa & Security Settings ---
const char *userKey = "6tfDs$wEq3!"; // Security key. Must match the receiver.
#define RF_FREQUENCY 915555000 // Hz. Must match the frequency of the receiver.
#define TX_OUTPUT_POWER 2     // dBm from 2 to 20. (14 is a safe max on battery)

// --- LoRa Radio Parameters ---
#define LORA_BANDWIDTH 0         // [0: 125 kHz]
#define LORA_SPREADING_FACTOR 7  // [SF7..SF12]
#define LORA_CODINGRATE 1        // [1: 4/5]
#define LORA_PREAMBLE_LENGTH 8   // Same for Tx and Rx
#define LORA_FIX_LENGTH_PAYLOAD_ON false
#define LORA_IQ_INVERSION_ON false

// --- Global Variables ---
SSD1306Wire oledDisplay(0x3c, 500000, SDA_OLED, SCL_OLED, GEOMETRY_128_64, RST_OLED);
Robojax_HeltecLoRa32 robojaxDisplay(&oledDisplay);

// This variable is required by the Robojax_HeltecLoRa32 library
bool lora_idle = true;

// State management for toggle switch
String displayTextStateValue = displayTextRelayToggleOFF;
bool toggleState = false; // false = OFF, true = ON

// Variables for button polling and debouncing
int buttonState;
int lastButtonState = HIGH; // The pull-up resistor keeps the pin HIGH
unsigned long lastDebounceTime = 0;
unsigned long debounceDelay = 50;


// --- Radio Event Functions ---
void OnTxDone(void) {
  if (debug) Serial.println("TX done.");
  lora_idle = true;
}

void OnTxTimeout(void) {
  Radio.Sleep(); // Put radio in low-power mode
  if (debug) Serial.println("TX Timeout.");
  lora_idle = true;
}

static RadioEvents_t RadioEvents;

// --- Setup Function ---
void setup() {
  Serial.begin(115200);
  
  // Power on external components
  pinMode(Vext, OUTPUT);
  digitalWrite(Vext, LOW);
  delay(100);

  // Initialize OLED display
  robojaxDisplay.begin();
  
  // Initialize the push button pin.
  // The Heltec board has a 10k pull-up on pin 0, so we use INPUT mode.
  pinMode(PUSH_BUTTON_PIN, INPUT);

  // Initialize LoRa Radio
  Mcu.begin(HELTEC_BOARD, SLOW_CLK_TPYE);
  RadioEvents.TxDone = OnTxDone;
  RadioEvents.TxTimeout = OnTxTimeout;
  Radio.Init(&RadioEvents);
  Radio.SetChannel(RF_FREQUENCY);
  Radio.SetTxConfig(MODEM_LORA, TX_OUTPUT_POWER, 0, LORA_BANDWIDTH,
                    LORA_SPREADING_FACTOR, LORA_CODINGRATE,
                    LORA_PREAMBLE_LENGTH, LORA_FIX_LENGTH_PAYLOAD_ON,
                    true, 0, 0, LORA_IQ_INVERSION_ON, 3000);
  
  if (debug) Serial.println("Toggle TX (Polling Version) Initialized.");
}

// --- Main Loop ---
void loop() {
  // 1. Read the button state with debouncing
  int reading = digitalRead(PUSH_BUTTON_PIN);

  // If the switch changed, due to noise or pressing, reset the debounce timer
  if (reading != lastButtonState) {
    lastDebounceTime = millis();
  }

  if ((millis() - lastDebounceTime) > debounceDelay) {
    // If the button state has been stable for long enough, and it's different
    if (reading != buttonState) {
      buttonState = reading;

      // We only act on the PRESS, which is when the state changes to LOW
      if (buttonState == LOW) {
        if (debug) Serial.println("Button press detected.");
        
        // Flip the toggle state
        toggleState = !toggleState;
        
        // Update the text to be displayed and sent
        if (toggleState) {
          displayTextStateValue = displayTextRelayToggleON;
        } else {
          displayTextStateValue = displayTextRelayToggleOFF;
        }
        
        if (debug) Serial.printf("New state: %s. Attempting to transmit...\n", displayTextStateValue.c_str());
        
        // Transmit the new state immediately
        robojaxDisplay.sendStringSecure(displayTextStateValue);
        
        if (debug) Serial.println("Transmission call finished.");
      }
    }
  }
  
  // Save the reading for the next loop
  lastButtonState = reading;

  // 2. Update the OLED Display continuously
  robojaxDisplay.displayLineText(displayTextTitle, 0, 0, 24, false);
  robojaxDisplay.displayText(displayTextTX, 127, 5, 10, TEXT_ALIGN_RIGHT);
  robojaxDisplay.displayLineText(displayTextStateValue, 0, 30, 24, true);

  // Let the ESP32 handle radio events
  Radio.IrqProcess();

  delay(10); // A small delay to keep things stable
}
