/*
 * file name: Relay_Secure_TX_Momentary
 *written on July 03, 2025 by Ahamd Shamshiri www.Robojax.com
 * Description:
 * This code is for a LoRa transmitter using a Heltec WiFi LoRa 32 V3.
 * - When the push button (pin 0) is pressed and held, it transmits "ON".
 * - When the button is released, it transmits "OFF" for 1 second.
 * - This uses the onboard 10k pull-up resistor, so the button should connect pin 0 to GND.
 * Watch Full video instruction: https://youtu.be/4QZlciNqiDU
 * 📚⬇️ Download and resource page https://robojax.com/T641
 */
#include <Wire.h>
#include <HT_SSD1306Wire.h>
#include "LoRaWan_APP.h"
#include <Robojax_HeltecLoRa32.h>

// --- Configuration ---
bool debug = true; // Set to true to see Serial Monitor messages
#define PUSH_BUTTON_PIN 0  // The pin connected to the push button

// --- Display & Payload Text ---
const char *displayTexttitle = "Relay:";
const char *displayTextTX = "(TX)";
const char *displayTextRelayON = "ON";
const char *displayTextRelayOFF = "OFF";
const int transmitDelayTime = 1000; // How long to transmit OFF after release (in ms)

// --- LoRa & Security Settings ---
const char *userKey = "6tfDs$wEq3!"; // Security key. Must match the receiver.
#define RF_FREQUENCY 915555000 // Hz. Must match the frequency of the receiver.
#define TX_OUTPUT_POWER 2     // dBm from 2 to 20. (14 is a safe max on battery)

// --- LoRa Radio Parameters ---
#define LORA_BANDWIDTH 0         // [0: 125 kHz]
#define LORA_SPREADING_FACTOR 7  // [SF7..SF12]
#define LORA_CODINGRATE 1        // [1: 4/5]
#define LORA_PREAMBLE_LENGTH 8   // Same for Tx and Rx
#define LORA_FIX_LENGTH_PAYLOAD_ON false
#define LORA_IQ_INVERSION_ON false

// --- Global Variables ---
SSD1306Wire oledDisplay(0x3c, 500000, SDA_OLED, SCL_OLED, GEOMETRY_128_64, RST_OLED);
Robojax_HeltecLoRa32 robojaxDisplay(&oledDisplay);

// Initial state is HIGH because of the hardware pull-up resistor
int currentButtonState = HIGH;

// This variable is required by the Robojax_HeltecLoRa32 library
bool lora_idle = true;

// Interrupt and timing variables
volatile bool buttonPressed = false;
unsigned long lastInterruptTime = 0;
const unsigned long debounceDelay = 50; // 50ms debounce time

// State management for momentary switch
String displayTextStateValue = displayTextRelayOFF;
bool shouldTransmitOff = false;
unsigned long lastTransmitTime = 0;

// --- Radio Event Functions ---
void OnTxDone(void) {
  if (debug) Serial.println("TX done.");
  lora_idle = true;
}

void OnTxTimeout(void) {
  Radio.Sleep(); // Put radio in low-power mode
  if (debug) Serial.println("TX Timeout.");
  lora_idle = true;
}

static RadioEvents_t RadioEvents;

// --- Interrupt Service Routine ---
void IRAM_ATTR handleButtonInterrupt() {
  if (millis() - lastInterruptTime > debounceDelay) {
    buttonPressed = true;
    lastInterruptTime = millis();
  }
}

// --- Setup Function ---
void setup() {
  Serial.begin(115200);
  
  // Power on external components
  pinMode(Vext, OUTPUT);
  digitalWrite(Vext, LOW);
  delay(100);

  // Initialize OLED display
  robojaxDisplay.begin();
  
  // Initialize the push button pin.
  // The Heltec board has a 10k pull-up on pin 0, so we use INPUT mode.
  pinMode(PUSH_BUTTON_PIN, INPUT);
  attachInterrupt(digitalPinToInterrupt(PUSH_BUTTON_PIN), handleButtonInterrupt, CHANGE);

  // Initialize LoRa Radio
  Mcu.begin(HELTEC_BOARD, SLOW_CLK_TPYE);
  RadioEvents.TxDone = OnTxDone;
  RadioEvents.TxTimeout = OnTxTimeout;
  Radio.Init(&RadioEvents);
  Radio.SetChannel(RF_FREQUENCY);
  Radio.SetTxConfig(MODEM_LORA, TX_OUTPUT_POWER, 0, LORA_BANDWIDTH,
                    LORA_SPREADING_FACTOR, LORA_CODINGRATE,
                    LORA_PREAMBLE_LENGTH, LORA_FIX_LENGTH_PAYLOAD_ON,
                    true, 0, 0, LORA_IQ_INVERSION_ON, 3000);
  
  if (debug) Serial.println("Momentary TX Initialized. Waiting for button press...");
}

// --- Main Loop ---
void loop() {
  // 1. Check if the button state has changed
  if (buttonPressed) {
    buttonPressed = false; // Reset interrupt flag
    int newState = digitalRead(PUSH_BUTTON_PIN);

    // Only act if the state has physically changed
    if (newState != currentButtonState) {
      currentButtonState = newState; // Update the current state

      if (currentButtonState == LOW) { // Button was just pressed
        displayTextStateValue = displayTextRelayON;
        shouldTransmitOff = false; // Stop any pending OFF transmissions
        if (debug) Serial.println("Button Pressed -> Transmitting ON");
      } else { // Button was just released
        displayTextStateValue = displayTextRelayOFF;
        shouldTransmitOff = true; // Enable the OFF transmission period
        lastTransmitTime = millis(); // Start the timer
        if (debug) Serial.println("Button Released -> Transmitting OFF for a short duration");
      }
    }
  }

  // 2. Update the OLED Display
  robojaxDisplay.displayLineText(displayTexttitle, 0, 0, 24, false);
  robojaxDisplay.displayText(displayTextTX, 127, 5, 10, TEXT_ALIGN_RIGHT);
  robojaxDisplay.displayLineText(displayTextStateValue, 0, 30, 24, true);

  // 3. Handle Transmission Logic
  // Condition 1: Transmit continuously if button is held down (state is LOW)
  if (currentButtonState == LOW) {
    robojaxDisplay.sendStringSecure(displayTextStateValue);
  }
  // Condition 2: Transmit OFF for a short period after release
  else if (shouldTransmitOff) {
    if (millis() - lastTransmitTime < transmitDelayTime) {
      robojaxDisplay.sendStringSecure(displayTextStateValue);
    } else {
      shouldTransmitOff = false; // Timer has expired, stop transmitting
    }
  }

  delay(100);
}
