#include "Robojax_HeltecLoRa32.h"
#include <HT_SSD1306Wire.h> 
#include "LoRaWan_APP.h"
#include <mbedtls/aes.h>
#include <cstring>

extern volatile bool lora_idle; // declare if it's set in LoRaWan_APP
extern bool debug;
extern const char *userKey;


#define BATTERY_PIN 1
#define ADC_CTRL_PIN 37
#define CALIB_FACTOR 1.075f// used for internal battery
#define DIVIDER_RATIO 4.9f//do not change
#define ADC_REF 3.3f
#define ADC_MAX 4095


// Default constructor (no display)
Robojax_HeltecLoRa32::Robojax_HeltecLoRa32() {
    display = nullptr;
}

// Constructor with display pointer
Robojax_HeltecLoRa32::Robojax_HeltecLoRa32(SSD1306Wire* externalDisplay) {
    display = externalDisplay;
}

// Optional setter method (in case display is assigned later)
void Robojax_HeltecLoRa32::setDisplay(SSD1306Wire* disp) {
    display = disp;
}

// Initializes the display
void Robojax_HeltecLoRa32::begin() {
    if (display != nullptr) {
        display->init();                     // Initialize OLED
        display->setFont(ArialMT_Plain_10);  // Default font
    }
}

void Robojax_HeltecLoRa32::clear() {
    display->clear();

}

void Robojax_HeltecLoRa32::calculatResistor(int R2, float voltage)
{
    Serial.print("Calculating R1 when R2 is :");
    Serial.print(R2);
    Serial.print(" Ohms and Maximum Input Voltage is ");  
    Serial.print(voltage);  
    Serial.println("V");  
    Serial.print("***** R1 Should Be ");  
    float R1 = (voltage - ADC_REF)/ (ADC_REF/ (float)R2);
    Serial.print(R1);  
    Serial.println(" Ohms");  
    while(1);
}

/*
prints maximum volage
*/
void Robojax_HeltecLoRa32::maxVoltage(int R1, int R2)
{
  float maxVoltage = ( ADC_REF)  * (1 + (float)R1/(float)R2);
    Serial.print("****Maximum Voltage: ");
    Serial.print(maxVoltage);
    Serial.println("V");

}//maxVoltage() end


void Robojax_HeltecLoRa32::sendString(String str) {

    if (lora_idle == true) {
        delay(1000); // optional delay
        Radio.Send((uint8_t *)str.c_str(), str.length());
        if(debug)
        {
            Serial.print("Sending: ");
            Serial.println(str);
        }
        lora_idle = false;
    }

    Radio.IrqProcess();
}

void Robojax_HeltecLoRa32::sendStringSecure(String str) {

    if (lora_idle == true) {
        delay(1000); // optional delay
        uint8_t data[32];       
        memset(data, 0, sizeof(data));  // Zero-padding
        strncpy((char*)data, str.c_str(), sizeof(data) - 1); // Copy string safely
        encryptAES(data, userKey);  // Encrypt before sending 
        
        Radio.Send(data,  sizeof(data));
        if(debug)
        {
            Serial.print("Sending: ");
            Serial.println((char *)data);
        }
        lora_idle = false;
    }

    Radio.IrqProcess();
}



void Robojax_HeltecLoRa32::sendBatteryVoltage() {
    float voltage = readInternalBattery(); // uses your own method
    String stringVoltage = String(voltage, 2); // format nicely

    if (lora_idle == true) {
        delay(1000); // optional delay
        Radio.Send((uint8_t *)stringVoltage.c_str(), stringVoltage.length());
        if(debug){
            Serial.print("Sending: ");
            Serial.println(stringVoltage);
        }
        lora_idle = false;
    }

    Radio.IrqProcess();
}

float Robojax_HeltecLoRa32::readAnyVoltage(int adcPin,float R1, float R2) {
 

    uint32_t voltage_mV =0;
    for (int i = 0; i < 16; i++) {
        voltage_mV += analogReadMilliVolts(adcPin);
        delay(1);
    }
    voltage_mV /= 16;
    if(debug){
        Serial.print("voltage_mV: ");
        Serial.println(voltage_mV);
    }
   float  Voltage = (((float) voltage_mV) / 1000.0)  * (1 + (float)R1/(float)R2);

    return Voltage;
}


float Robojax_HeltecLoRa32::readInternalBattery() {
    // Enable ADC control pin
    pinMode(ADC_CTRL_PIN, OUTPUT);
    digitalWrite(ADC_CTRL_PIN, HIGH);
    delay(10); // let it stabilize

    int adcValue = 0;
    for (int i = 0; i < 16; i++) {
        adcValue += analogRead(BATTERY_PIN);
    }
    adcValue /= 16;

    interBatteryVoltage = adcValue * (ADC_REF / ADC_MAX) * DIVIDER_RATIO * CALIB_FACTOR;

    return interBatteryVoltage;
}

void Robojax_HeltecLoRa32::displayLargeBattery(float voltage, int type) {

    display->clear();
    
    // Calculate percentage (3.2V-4.2V range)
    int percent = map(constrain(voltage*100, 320, 420), 320, 420, 0, 100);
  
    // Set large font for percentage
    display->setFont(ArialMT_Plain_24);
    display->setTextAlignment(TEXT_ALIGN_CENTER);
    
    if(type ==1){
    // Big percentage in center (Y=10)
    display->drawString(64, 0, String(percent) + "%");
    }else{
     display->drawString(64, 0, String(voltage) + "V"); 
    }
  
    // Large battery outline (centered, width=80, height=30)
    int battWidth = 80;
    int battHeight = 30;
    int battX = (128 - battWidth)/2;
    int battY = 30;
    
    // Battery outline
    display->drawRect(battX, battY, battWidth, battHeight);
    display->fillRect(battX + battWidth, battY + 10, 5, 10); // Tip
  
    // Filled portion (with 2px margin)
    int fillWidth = map(percent, 0, 100, 0, battWidth - 4);
    display->fillRect(battX + 2, battY + 2, fillWidth, battHeight - 4);
  
    // Voltage text below (medium font)
    display->setFont(ArialMT_Plain_16);
    display->drawString(64, battY + battHeight + 5, String(voltage, 2) + "V");
  
    display->display();
  }

void Robojax_HeltecLoRa32::displaySmallBattery(float voltage) {
    
    //display->clear();  // Clear display before new content
    // LiPo Percentage (3.2V-4.2V range)
    int percent = map(constrain(voltage*100, 320, 420), 320, 420, 0, 100);
    
    // Battery Icon (Top-Right)
    display->setFont(ArialMT_Plain_10);
    display->setTextAlignment(TEXT_ALIGN_RIGHT);
    
    
    display->setColor(BLACK);
    display->fillRect(102, 4, 22, 20); // Width = display width minus x
    display->setColor(WHITE);
    display->display();

    // Icon
    display->drawRect(100, 2, 22, 10);    // Outline
    display->fillRect(122, 4, 2, 6);      // Tip
    display->fillRect(102, 4, map(percent, 0, 100, 0, 18), 6); // Fill
    
    // Text
    // display->drawString(98, 2, String(percent) + "%");
    // display->drawString(98, 12, String(voltage,2) + "V");
      display->display();  // Update OLED
}



void Robojax_HeltecLoRa32::displayText(String text, int x, int y, int fontSize, int align, bool clear) {
    if (!display) return;

    if(clear)display->clear();

    // ✅ Cast int to correct enum inside .cpp
    display->setTextAlignment((DISPLAY_TEXT_ALIGNMENT)align);

    switch (fontSize) {
        case 10: display->setFont(ArialMT_Plain_10); break;
        case 16: display->setFont(ArialMT_Plain_16); break;
        case 24: display->setFont(ArialMT_Plain_24); break;
        default: display->setFont(ArialMT_Plain_10); break;
    }

    display->drawString(x, y, text);
    display->display();
}


void Robojax_HeltecLoRa32::displayCenteredText(String text, int y, int fontSize) {
    if (!display) return;

    switch (fontSize) {
        case 10: display->setFont(ArialMT_Plain_10); break;
        case 16: display->setFont(ArialMT_Plain_16); break;
        case 24: display->setFont(ArialMT_Plain_24); break;
        default: display->setFont(ArialMT_Plain_16); break;
    }

    int textWidth = display->getStringWidth(text);
    int x = (128 - textWidth) / 2; // 128 is display width

    display->clear();
    display->setTextAlignment(TEXT_ALIGN_LEFT);
    display->drawString(x, y, text);
    display->display();
}

void Robojax_HeltecLoRa32::clearLineArea(int x, int y, int fontSize) {
    if (!display) return;

    int height = fontSize;
    display->setColor(BLACK);
    display->fillRect(x, y, 128 - x, height); // Width = display width minus x
    display->setColor(WHITE);
    display->display();
}

void Robojax_HeltecLoRa32::displayLineText(String text, int x, int y, int fontSize, bool clear) {
    if (!display) return;

    setFontSize(fontSize);
    if(clear)
    {
        clearLineArea(x, y, fontSize);
    }
    display->setTextAlignment(TEXT_ALIGN_LEFT);
    display->drawString(x, y, text);
    display->display();
}

void Robojax_HeltecLoRa32::displayClearLine(int y, int fontSize) {
    if (!display) return;

    int height;
    setFontSize(fontSize);

    display->setColor(BLACK); // "Draw" in black to clear the line
    display->fillRect(0, y, 128, height); // Clear a horizontal band
    display->setColor(WHITE); // Reset back to white for normal drawing
    display->display(); // Update the display
}




void Robojax_HeltecLoRa32::setFontSize(int fontSize) {
    if (!display) return;

    switch (fontSize) {
        case 10: display->setFont(ArialMT_Plain_10); break;
        case 16: display->setFont(ArialMT_Plain_16); break;
        case 24: display->setFont(ArialMT_Plain_24); break;
        default: display->setFont(ArialMT_Plain_10); break;
    }
}



void Robojax_HeltecLoRa32::processKey(const char *userKey, uint8_t *processedKey, size_t keySize) {
    memset(processedKey, 0, keySize);
    size_t len = strlen(userKey);
    if (len > keySize) len = keySize;
    memcpy(processedKey, userKey, len);
}

void Robojax_HeltecLoRa32::encryptAES(uint8_t *data, const char *key) {
    mbedtls_aes_context aes;
    uint8_t processedKey[16];
    processKey(key, processedKey, 16);

    mbedtls_aes_init(&aes);
    mbedtls_aes_setkey_enc(&aes, processedKey, 128);
    mbedtls_aes_crypt_ecb(&aes, MBEDTLS_AES_ENCRYPT, data, data);
    mbedtls_aes_free(&aes);
}

void Robojax_HeltecLoRa32::decryptAES(uint8_t *data, const char *key) {
    mbedtls_aes_context aes;
    uint8_t processedKey[16];
    processKey(key, processedKey, 16);

    mbedtls_aes_init(&aes);
    mbedtls_aes_setkey_dec(&aes, processedKey, 128);
    mbedtls_aes_crypt_ecb(&aes, MBEDTLS_AES_DECRYPT, data, data);
    mbedtls_aes_free(&aes);
}
